/*global define */

// Motion Trigger behavior for AE World; by David Simons <dsimons@adobe.com>

define(["lib/tasks", "src/math/Mat3"],
  function (tasks, mat3) {
	"use strict";
				 
	function setupTriggerableLayer (args, paramID)
	{
		var bHideSiblings = true;

		var aLayers = args.getStaticParam(paramID);

		aLayers.forEach(function (lay) {
			lay.setTriggerable(bHideSiblings);
		});
	}
	  
	function defineLayer (id, dx, dy, uiName, match, uiToolTip0)
	{
		var def = { 
				id: id, type: "layer", uiName: uiName,
				dephault: { match: "//" + match },
				customData: { dx: dx, dy: dy }	// TODO: make this official behavior-specific field
		};
		
		if (uiToolTip0) {
			def.uiToolTip = uiToolTip0;
		}
		
		return def;
	}
	
	var aLayerTags = [
		{
			id: "Adobe.MotionTrigger.AtRest",
			artMatches: ["at rest"],	// TODO: add language synonyms
			uiName: "$$$/animal/Behavior/MotionTrigger/UIName/AtRest=At Rest",
			tagType: "layertag",
			uiGroups: [{id: "Adobe.TagGroup.MotionTrigger", subSort: 0}]
		},					
		{
			id: "Adobe.MotionTrigger.MovingLeft",
			artMatches: ["moving left"],
			uiName: "$$$/animal/Behavior/MotionTrigger/UIName/MovingLeft=Moving Left",
			tagType: "layertag",
			uiGroups: [{id: "Adobe.TagGroup.MotionTrigger", subSort: 0}]
		},					
		{
			id: "Adobe.MotionTrigger.MovingRight",
			artMatches: ["moving right"],
			uiName: "$$$/animal/Behavior/MotionTrigger/UIName/MovingRight=Moving Right",
			tagType: "layertag",
			uiGroups: [{id: "Adobe.TagGroup.MotionTrigger", subSort: 0}]
		},					
		{
			id: "Adobe.MotionTrigger.MovingUp",
			artMatches: ["moving up"],
			uiName: "$$$/animal/Behavior/MotionTrigger/UIName/MovingUp=Moving Up",
			tagType: "layertag",
			uiGroups: [{id: "Adobe.TagGroup.MotionTrigger", subSort: 0}]
		},					
		{
			id: "Adobe.MotionTrigger.MovingDown",
			artMatches: ["moving down"],
			uiName: "$$$/animal/Behavior/MotionTrigger/UIName/MovingDown=Moving Down",
			tagType: "layertag",
			uiGroups: [{id: "Adobe.TagGroup.MotionTrigger", subSort: 0}]
		},
		{
			id: "Adobe.MotionTrigger.InTransition",
			artMatches: ["in transition"],
			uiName: "$$$/animal/Behavior/MotionTrigger/UIName/InTransition=In Transition",
			tagType: "layertag",
			uiGroups: [{id: "Adobe.TagGroup.MotionTrigger", subSort: 0}]
		}				
	];

	
	var aLayerParamDefs = [
		defineLayer("atRest", 0, 0,
					"$$$/animal/Behavior/MotionTrigger/LayerParam/AtRest=At Rest",
					 "Adobe.MotionTrigger.AtRest",
					 "$$$/animal/Behavior/MotionTrigger/LayerParam/AtRest/ToolTip=Layers that will be triggered when the puppet isn^}t moving"),
		defineLayer("movingLeft", -1, 0,
					"$$$/animal/Behavior/MotionTrigger/LayerParam/MovingLeft=Moving Left",
					 "Adobe.MotionTrigger.MovingLeft",
					 "$$$/animal/Behavior/MotionTrigger/LayerParam/MovingLeft/ToolTip=Layers that will be triggered when the puppet is moving to the left"),
		defineLayer("movingRight", +1, 0,
					"$$$/animal/Behavior/MotionTrigger/LayerParam/MovingRight=Moving Right",
					 "Adobe.MotionTrigger.MovingRight",
					 "$$$/animal/Behavior/MotionTrigger/LayerParam/MovingRight/ToolTip=Layers that will be triggered when the puppet is moving to the right"),
		defineLayer("movingUp", 0, -1,
					"$$$/animal/Behavior/MotionTrigger/LayerParam/MovingUp=Moving Up",
					 "Adobe.MotionTrigger.MovingUp",
					 "$$$/animal/Behavior/MotionTrigger/LayerParam/MovingUp/ToolTip=Layers that will be triggered when the puppet is moving up"),
		defineLayer("movingDown", 0, +1,
					"$$$/animal/Behavior/MotionTrigger/LayerParam/MovingDown=Moving Down",
					 "Adobe.MotionTrigger.MovingDown",
					 "$$$/animal/Behavior/MotionTrigger/LayerParam/MovingDown/ToolTip=Layers that will be triggered when the puppet is moving down"),
		defineLayer("inTransition", 0, 0,
					"$$$/animal/Behavior/MotionTrigger/LayerParam/InTransition=In Transition",
					 "Adobe.MotionTrigger.InTransition",
					 "$$$/animal/Behavior/MotionTrigger/LayerParam/InTransition/ToolTip=Layers that will be triggered when the puppet switches from moving in one direction to another direction")
	];
	
	return {
		about:			"Motion Trigger, copyright 2015.",
		description:	"$$$/animal/Behavior/MotionTrigger/Desc=Switch between different layers based on the puppet^}s direction of movement",
		uiName:  		"$$$/animal/Behavior/MotionTrigger=Motion Trigger",
		
		defineTags: function () {
			return {aTags: aLayerTags};
		},
		
		defineParams : function () {
			var aParams = [];
			
			aLayerParamDefs.forEach(function (def) {
				aParams.push(def);
			});

			aParams.push({
				id: "threshold", type: "slider",
				uiName: "$$$/animal/Behavior/MotionTrigger/Param/Threshold=Speed Threshold",
				uiUnits: "$$$/animal/Behavior/MotionTrigger/Param/Threshold/units=px/frame",
				min: 0, max: 1000, dephault: 3,
				uiToolTip: "$$$/animal/Behavior/MotionTrigger/Param/Threshold/ToolTip=Speeds below this threshold will show the At Rest layer"
			});
			
			aParams.push({
				id: "minDuration", type: "slider",
				uiName: "$$$/animal/Behavior/MotionTrigger/Param/MinimumDuration=Minimum Duration",
				uiUnits: "$$$/animal/Behavior/MotionTrigger/Param/MinimumDuration/units=frames",
				min: 1, max: 240, dephault: 2,
				uiToolTip: "$$$/animal/Behavior/MotionTrigger/Param/MinimumDuration/ToolTip=Minimum number of frames to trigger a layer"
			});
			
			return aParams;
		},

		onCreateBackStageBehavior : function (/*self*/) {
			return { order: 1.0, importance : 0.0 };
		},

		onCreateStageBehavior : function (self, args) {
			aLayerParamDefs.forEach(function (def) {
				setupTriggerableLayer(args, def.id);
			});
			self.repeatedFramesShown = 1000;		// sentinel, just needs to be > minDuration max
			self.lastTriggered = "atRest";
		},

		onAnimate : function (self, args) {
			var layerToShow = "atRest",
				threshold = Math.max(args.getParam("threshold"), 0.01),
				minDuration = args.getParam("minDuration"),
				handle = args.stageLayer.privateLayer.getHandleTreeRoot(),
				x, y;
			
			var matScene_Layer = args.getLayerMatrixRelativeToScene(handle.getWarperLayer());
			var mat = mat3.multiply(matScene_Layer, tasks.handle.getFrameRelativeToLayer(handle));
			
			x = mat[6];		// pull horizontal offset component from matrix
			y = mat[7];

			if (self.repeatedFramesShown + 1 < minDuration) {
				// previously triggered layer hasn't been showing for long enough yet
				//	so show it again
				layerToShow = self.lastTriggered;
			} else {
				var dx, dy, f, maxf = -1;
				if (self.lastX !== undefined) {
					dx = x - self.lastX;
					dy = y - self.lastY;

					aLayerParamDefs.forEach(function (def) {
						var defdx = def.customData.dx, defdy = def.customData.dy;

						if (defdx || defdy) {
							f = defdx * dx + defdy * dy;
							if (f >= threshold && f > maxf && args.getParam(def.id).length > 0) {
								layerToShow = def.id;
								maxf = f;
							}
						}
					});
				}
				if (args.getParam("inTransition").length > 0 &&	// In Transition exists
						layerToShow !== self.lastTriggered &&	// change just happened
						layerToShow !== "atRest" &&				// not changing to rest
						self.lastTriggered !== "inTransition" &&	// not coming from transition already
						self.lastTriggered !== "atRest") {			// not coming from rest
					// then we're changing from one _moving_ view to another
					layerToShow = "inTransition"; // so show it
				}
			}
			
			var aLayers = args.getParam(layerToShow),
				priority = 0.2;	// more than Cycle Layers (in particular its sustainTrigger())
			aLayers.forEach(function (lay) {
				lay.trigger(priority); // will show this one and hide its untriggered siblings
			});
			
			if (layerToShow === self.lastTriggered) {
				self.repeatedFramesShown++;
			} else {
				self.repeatedFramesShown = 0;
				self.lastTriggered = layerToShow;
			}
			
			self.lastX = x;
			self.lastY = y;
		}
	};
});
